#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "thpmd.h"

typedef struct {
	u16 magic;
	u8  key;
	u8  name[13];
	u32 datalen;
	u32 filelen;
	u32 offset;
/*	u8  padding[4]; */
} th_file_entry;

int th_unrle(const u8 *in, u32 insize, u8 *out, u32 outsize)
{
	u8 ppr, pr, i;
	u32 inpos, outpos;

	pr = out[0] = in[0];
	for(inpos = outpos = 1; inpos < insize && outpos < outsize;) {
		ppr = pr;
		pr = out[outpos++] = in[inpos++];
		if(ppr == pr) {
			for (i = 0; i < in[inpos] && outpos < outsize; ++i)
				out[outpos++] = pr;
			++inpos;
		}
	}
	return (inpos == insize);
}

int th_extract(FILE *fi, th_file_entry *entry)
{
	FILE *fo;
	char *file;
	int i;

	printf("Extracting %s...\n", entry->name);

	if(fseek(fi, entry->offset, SEEK_SET) == -1) {
		perror("fseek");
		return 0;
	}

	file = malloc(entry->datalen);
	if(!file) {
		perror("malloc");
		return 0;
	}
	
	if(fread(file, 1, entry->datalen, fi) != entry->datalen) {
		if(feof(fi)) {
			fprintf(stderr, "THPMD: Data seems incomplete\n");
		} else {
			perror("fread");
		}
		free(file);
		return 0;
	}

	/* f[^𕜌 */
	for(i = 0; i < entry->datalen; ++i) file[i] ^= 0x12;

	/* KvɉRLEWJ */
	if(entry->datalen < entry->filelen) {
		char *plain = malloc(entry->filelen);
		if(!plain) {
			perror("malloc");
			free(file);
			return 0;
		}
		if(!th_unrle(file, entry->datalen, plain, entry->filelen)) {
			fprintf(stderr, "THPMD: Failed to uncompress data");
			free(file);
			free(plain);
			return 0;
		}
		free(file);
		file = plain;
	}

	/* f[^t@Cɏo */
	fo = fopen(entry->name, "wb");
	if(fo == NULL) {
		perror("fopen");
		free(file);
		return 0;
	}
	if(fwrite(file, 1, entry->filelen, fo) != entry->filelen) {
		perror("fwrite");
		fclose(fo);
		free(file);
		return 0;
	}

	fclose(fo);
	free(file);
	return 1;
}

int th_readentry(FILE *fp, th_file_entry *entry)
{
	u8 buf[0x20];
	if(fread(&buf, 1, 0x20, fp) != 0x20) return 0;
	entry->magic = to16le(buf);
	entry->key = *(buf + 2);
	memcpy(entry->name, buf + 3, 13);
	entry->datalen = to32le(buf + 0x10);
	entry->filelen = to32le(buf + 0x14);
	entry->offset = to32le(buf + 0x18);
	return 1;
}

int th_chkentry(const th_file_entry *entry, long filelen)
{
	if(entry->magic != 0x9595 && entry->magic != 0xF388) return 0;
	if(entry->offset >= filelen) return 0;
	if(entry->datalen == 0 || entry->filelen == 0) return 0;
	if(filelen - entry->offset < entry->datalen) return 0;
	return 1;
}

int th_chkname(const char *name)
{
	int i, j;
	/* t@C8.3`łȂƃ_ */
	for(i = j = 0; i < 8 && isfchr(name[i]); ++i);
	if(name[i] == '.')
		for(j = 1; j < 4 && isfchr(name[i + j]); ++j);
	return (name[i + j] == '\0') && i && (j != 1);
}

int th_ispmd(const char *name)
{
	/* ̑ */
	char *ext = strchr(name, '.');
	if(!ext) return 0;
	return
		!strcmp(++ext, "M") || !strcmp(ext, "M2") ||
		!strcmp(ext, "M26") || !strcmp(ext, "M86") ||
		!strcmp(ext, "MMD");
}

void usage()
{
	puts(
	"THPMD for TH02.\n"
	"\n"
	"Usage: THPMDB FILE"
	);
}

int main(int argc, char *argv[])
{
	FILE *fi;
	th_file_entry entry;
	long filelen;
	int i, tmpofs;

	if(argc < 2) {
		usage();
		return EXIT_SUCCESS;
	}

	fi = fopen(argv[1], "rb");
	if(!fi) {
		perror("open");
		return EXIT_FAILURE;
	}

	fseek(fi, 0, SEEK_END);
	filelen = ftell(fi);
	rewind(fi);

	for (;;) {
		/* t@CGgǂ */
		if(!th_readentry(fi, &entry)) {
			if(feof(fi)) {
				fprintf(stderr, "THPMD: File entries seem incomplete.");
			} else {
				perror("fread");
			}
			fclose(fi);
			return EXIT_FAILURE;
		}

		/* t@CGg̐ */
		if(entry.magic == 0) break;
		if(!th_chkentry(&entry, filelen)) {
			fprintf(stderr, "THPMD: File entry seems broken.");
			fclose(fi);
			return EXIT_FAILURE;
		}
		/* t@C𕜌 */
		for(i = 0; i < 12 && entry.name[i]; ++i) {
			entry.name[i] ^= 0xFF;
		}

		/* t@C̐ */
		if(!th_chkname(entry.name)) {
			fprintf(stderr, "THPMD: File name seems broken.");
			fclose(fi);
			return EXIT_FAILURE;
		}

		if(th_ispmd(entry.name)) {
			tmpofs = ftell(fi);
			if(!th_extract(fi, &entry)) {
				fclose(fi);
				return EXIT_FAILURE;
			}
			fseek(fi, tmpofs, SEEK_SET);
		}
	}

	puts("Finished extracting files.");

	fclose(fi);
	
	return EXIT_SUCCESS;
}
